document.addEventListener('DOMContentLoaded', function() {
    // Initialize all slideshow modules on the page
    const slideshows = document.querySelectorAll('.responsive-slideshow');
    
    slideshows.forEach(function(slideshow) {
        initSlideshow(slideshow);
    });
    
    function initSlideshow(slideshow) {
        const slides = slideshow.querySelectorAll('.slide');
        const container = slideshow.querySelector('.slideshow-container');
        const prevBtn = slideshow.querySelector('.prev-slide');
        const nextBtn = slideshow.querySelector('.next-slide');
        const indicators = slideshow.querySelectorAll('.indicator');
        const transition = slideshow.dataset.transition || 'fade';
        const duration = parseInt(slideshow.dataset.duration) || 800;
        
        let currentSlideIndex = 0;
        let isAnimating = false;
        
        // Set initial heights
        function setContainerHeight() {
            if (slides.length > 0) {
                const firstSlideHeight = slides[0].offsetHeight;
                container.style.minHeight = firstSlideHeight + 'px';
                
                // For fade transition, we need to make sure all slides maintain proper height
                if (transition === 'fade') {
                    slideshow.style.height = 'auto';
                    container.style.height = firstSlideHeight + 'px';
                }
            }
        }
        
        // Initial setup
        if (slides.length > 0) {
            // Make first slide active
            slides[0].classList.add('active');
            if (indicators.length > 0) {
                indicators[0].classList.add('active');
            }
            
            // Set initial heights after images have loaded
            if (slides[0].querySelector('img')) {
                const firstImg = slides[0].querySelector('img');
                if (firstImg.complete) {
                    setContainerHeight();
                } else {
                    firstImg.onload = setContainerHeight;
                }
            } else {
                setContainerHeight();
            }
        }
        
        // Listen for window resize to adjust heights
        window.addEventListener('resize', setContainerHeight);
        
        // Handle slide navigation
        function goToSlide(index, direction) {
            if (isAnimating || index === currentSlideIndex || slides.length <= 1) return;
            
            isAnimating = true;
            
            // Direction: 1 for next, -1 for prev
            direction = direction || (index > currentSlideIndex ? 1 : -1);
            
            // Get current and next slides
            const currentSlide = slides[currentSlideIndex];
            const nextSlide = slides[index];
            
            // Update container height to match the next slide
            if (nextSlide.offsetHeight > 0) {
                container.style.height = nextSlide.offsetHeight + 'px';
            }
            
            // Clear any previous classes
            slides.forEach(slide => {
                if (slide !== currentSlide && slide !== nextSlide) {
                    slide.classList.remove('active', 'prev');
                }
            });
            
            // Special handling for slide transition
            if (transition === 'slide') {
                // Set up initial positions based on direction
                if (direction > 0) {
                    // Going forward: next comes from right
                    nextSlide.classList.remove('prev');
                } else {
                    // Going backward: next comes from left
                    nextSlide.classList.add('prev');
                }
                
                // Force browser reflow to ensure transitions work
                void nextSlide.offsetWidth;
            }
            
            // Add active class to next slide
            nextSlide.classList.add('active');
            
            // Update indicators
            if (indicators.length > 0) {
                indicators[currentSlideIndex].classList.remove('active');
                indicators[index].classList.add('active');
            }
            
            // Remove active class from current slide after transition
            setTimeout(() => {
                currentSlide.classList.remove('active');
                currentSlideIndex = index;
                isAnimating = false;
                
                // Double-check height after transition
                if (transition === 'fade' && nextSlide.offsetHeight > 0) {
                    container.style.height = nextSlide.offsetHeight + 'px';
                }
            }, duration);
        }
        
        function nextSlide() {
            const next = (currentSlideIndex + 1) % slides.length;
            goToSlide(next, 1);
        }
        
        function prevSlide() {
            const prev = (currentSlideIndex - 1 + slides.length) % slides.length;
            goToSlide(prev, -1);
        }
        
        // Event listeners
        if (prevBtn) {
            prevBtn.addEventListener('click', function(e) {
                e.preventDefault();
                prevSlide();
            });
        }
        
        if (nextBtn) {
            nextBtn.addEventListener('click', function(e) {
                e.preventDefault();
                nextSlide();
            });
        }
        
        // Indicator buttons
        indicators.forEach((indicator, index) => {
            indicator.addEventListener('click', function() {
                const direction = index > currentSlideIndex ? 1 : -1;
                goToSlide(index, direction);
            });
        });
        
        // Keyboard navigation
        slideshow.addEventListener('keydown', function(e) {
            if (e.key === 'ArrowLeft') {
                prevSlide();
            } else if (e.key === 'ArrowRight') {
                nextSlide();
            }
        });
        
        // Make the slideshow focusable for keyboard navigation
        slideshow.setAttribute('tabindex', '0');
        
        // Touch swipe support
        let touchStartX = 0;
        let touchEndX = 0;
        
        slideshow.addEventListener('touchstart', function(e) {
            touchStartX = e.changedTouches[0].screenX;
        }, { passive: true });
        
        slideshow.addEventListener('touchend', function(e) {
            touchEndX = e.changedTouches[0].screenX;
            handleSwipe();
        }, { passive: true });
        
        function handleSwipe() {
            const swipeThreshold = 50;
            if (touchEndX < touchStartX - swipeThreshold) {
                // Swipe left - go to next slide
                nextSlide();
            } else if (touchEndX > touchStartX + swipeThreshold) {
                // Swipe right - go to previous slide
                prevSlide();
            }
        }
    }
});